let trackDetails = {
  search: "fate of",
  name: "The Fate of Ophelia",
  artist: "Taylor Swift",
  cover:
    "https://upload.wikimedia.org/wikipedia/en/f/f4/Taylor_Swift_%E2%80%93_The_Life_of_a_Showgirl_%28album_cover%29.png",
  colorPri: "hsl(86 40.66610541983196% 64.18819137254901%)",
  colorSec: "hsl(86 10.666105419831954% 9.18819137254902%)",
  colorCon: "#ffffff"
};
let isSearching = false;

const slider = document.querySelector("#slider");
const sliderThumb = slider.querySelector("#thumb");

const getSliderValue = () => {
  return parseFloat(slider.dataset.value || "0");
};

const onChange = (e) => {
  e?.preventDefault();
  let value = getSliderValue() / 100;
  if (e) {
    const sliderBounds = slider.getBoundingClientRect();
    value = (e.clientX - sliderBounds.x) / sliderBounds.width;
  }
  slider.style.setProperty("--value", value);
};

const onEnd = () => {
  window.removeEventListener("mousemove", onChange);
  window.removeEventListener("mouseup", onEnd);
};

const onStart = (e) => {
  window.addEventListener("mousemove", onChange);
  window.addEventListener("mouseup", onEnd);
  onChange(e);
};

slider.addEventListener("mousedown", onStart);
onChange();

const searchInputElement = document.querySelector("input");
const trackCardElement = document.querySelector("#track-card");
const trackNameElement = document.querySelector("#track-name");
const trackArtistElement = document.querySelector("#track-artist");
const trackLoaderElement = document.querySelector("#track-loader");
const onAlbumUpdate = () => {
  trackNameElement.innerText = trackDetails.name;
  trackArtistElement.innerText = trackDetails.artist;
  trackCardElement.style.setProperty(
    "--media-card-image",
    `url("${trackDetails.cover}")`
  );
  trackCardElement.style.setProperty(
    "--media-card-primary",
    trackDetails.colorPri
  );
  trackCardElement.style.setProperty(
    "--media-card-secondary",
    trackDetails.colorSec
  );
  // trackCardElement.style.setProperty('--media-card-contrast', trackDetails.colorCon);
  trackCardElement.style.setProperty("--media-card-contrast", "#ffffff");
};
async function extractColors(url) {
  const img = await new Promise((r) => {
    const i = new Image();
    i.crossOrigin = "anonymous";
    i.onload = () => r(i);
    i.src = url;
  });

  const c = document.createElement("canvas");
  const ctx = c.getContext("2d");
  c.width = img.width;
  c.height = img.height;
  ctx.drawImage(img, 0, 0);

  // Average colour
  const { data } = ctx.getImageData(0, 0, c.width, c.height);
  let r = 0,
    g = 0,
    b = 0;
  for (let i = 0; i < data.length; i += 4) {
    r += data[i];
    g += data[i + 1];
    b += data[i + 2];
  }
  r /= data.length / 4;
  g /= data.length / 4;
  b /= data.length / 4;

  // Convert to HSL
  const toHSL = (r, g, b) => {
    r /= 255;
    g /= 255;
    b /= 255;
    const max = Math.max(r, g, b),
      min = Math.min(r, g, b);
    let h,
      s,
      l = (max + min) / 2;

    if (max === min) {
      h = s = 0;
    } else {
      const d = max - min;
      s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
      switch (max) {
        case r:
          h = (g - b) / d + (g < b ? 6 : 0);
          break;
        case g:
          h = (b - r) / d + 2;
          break;
        case b:
          h = (r - g) / d + 4;
          break;
      }
      h /= 6;
    }
    return { h, s, l };
  };

  const { h, s, l } = toHSL(r, g, b);

  // Primary: lighter + slightly boosted saturation for uniqueness
  const primaryL = Math.min(l + 0.25, 1);
  const primaryS = Math.min(s + 0.2, 1);
  const primary = `hsl(${(h * 360) | 0} ${primaryS * 100}% ${primaryL * 100}%)`;

  // Secondary: darker + more neutral for legibility on top of primary
  const secondaryL = Math.max(l - 0.3, 0);
  const secondaryS = Math.max(s - 0.1, 0);
  const secondary = `hsl(${(h * 360) | 0} ${secondaryS * 100}% ${
    secondaryL * 100
  }%)`;

  // Black or white depending on entire image luminance
  const luminance = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
  const blackOrWhite = luminance > 0.5 ? "#000" : "#fff";

  return { primary, secondary, blackOrWhite };
}

const searchForSong = async (s) => {
  if (trackDetails?.search === s || isSearching) return;
  isSearching = true;
  trackLoaderElement.classList.toggle("hidden", false);
  try {
    const response = await fetch(
      `https://itunes.apple.com/search?entity=song&term=${s}`
    );
    const data = await response.json();
    console.log(data);
    const songData = (data?.results ?? [])[0];

    const songName = songData?.trackName ?? "Unknown";
    const songArtist = songData?.artistName ?? "Unknown";
    const songCover = (songData?.artworkUrl100 ?? "").replace(
      "100x100",
      "250x250"
    );
    const songColors = await extractColors(songCover);

    if (songData?.artistName) {
      trackDetails = {
        search: s,
        name: songName,
        artist: songArtist,
        cover: songCover,
        colorPri: songColors.primary,
        colorSec: songColors.secondary,
        colorCon: songColors.blackOrWhite
      };
      onAlbumUpdate();
    }
  } catch {}
  isSearching = false;
  trackLoaderElement.classList.toggle("hidden", true);
};
searchInputElement.addEventListener("blur", async () => {
  searchForSong(searchInputElement.value);
});
searchInputElement.addEventListener("keydown", async (e) => {
  if (e.key === "Enter") {
    searchForSong(searchInputElement.value);
  }
});