// Utils https://assets.codepen.io/573855/utils-v3.js

gsap.registerPlugin(ScrollTrigger, ScrollSmoother);

ScrollTrigger.config({
    limitCallbacks: true,
    ignoreMobileResize: true,
    autoRefreshEvents: 'DOMContentLoaded,load',
});

const scroller = (() => {
    if (typeof gsap === 'undefined' || typeof ScrollSmoother === 'undefined' || utils.device.isTouch()) {
        document.body.classList.add('normalize-scroll');
        return null;
    }

    return {
        initialize: (contentSelector = '.content-scroll', wrapperSelector = '.viewport-wrapper') =>
            ScrollSmoother.create({
                content: contentSelector,
                wrapper: wrapperSelector,
                smooth: 2,
                effects: false,
                normalizeScroll: true,
                preventDefault: true,
            }),
    };
})();

const createScrub = () => {

    let DOM = {
        container: null,
        pinContainer: null,
        triggerElement: null,
        imageLayer: null,
        overlayLayer: null,
        preloaderLayer: null,
        keyholeMask: null,
        corners: {
            topLeft: null,
            topRight: null,
            bottomLeft: null,
            bottomRight: null,
        },
        trackElement: null,
        trackMarkersContainer: null,
    };

    let observer = null;
    let scrollTrigger = null;

    let imageItems = [];
    let overlayCueItems = [];

    let isScrollLocked = true;
    let isImageUpdateQueued = false;

    let scrollDirection = 'down';
    let activeImageIndex = -1;
    let lastResolvedImageIndex = -1;
    let lastScrollProgress = 0;
    let queuedImageProgress = 0;

    let preloaderRemoved = false;
    let preloaderRemovalTick = null;

    let timelineDuration = 1;
    let lastRenderedTrackHeight = null;

    let lastAnimatedClipPath = '';

    const options = {
        selector: null,
        trigger: null,
        startOffset: null,
        duration: null,
        scrollFactor: 1,
    };

    const resolveTimelineDuration = (value, containerEl) => {
        const fromOptions = parseFloat(value);
        const fromAttr = parseFloat(containerEl.getAttribute('data-duration'));

        if (Number.isFinite(fromOptions) && fromOptions > 0) return fromOptions;
        if (Number.isFinite(fromAttr) && fromAttr > 0) return fromAttr;

        console.warn('[createScrub] No valid duration found; using default of 1s.');
        return 1;
    };

    const initialize = (opts = {}) => {
        reset();
        Object.assign(options, opts);

        const container = utils.dom.resolveElement(options.selector);
        if (!container) {
            console.warn('[createScrub] Invalid or missing selector.');
            return;
        }

        DOM.container = container;
        DOM.pinContainer = container.querySelector('.seq-pin-container');
        if (!DOM.pinContainer) {
            console.warn('[createScrub] Missing .seq-pin-container');
            return;
        }

        DOM.triggerElement = container.querySelector(options.trigger) || DOM.pinContainer;
        DOM.imageLayer = container.querySelector('.seq-img-layer');
        DOM.overlayLayer = container.querySelector('.seq-overlay-layer');
        DOM.preloaderLayer = container.querySelector('.seq-preloader-layer');
        DOM.keyholeMask = container.querySelector('.seq-keyhole');
        DOM.corners = {
            topLeft: DOM.pinContainer.querySelector('.top-left'),
            topRight: DOM.pinContainer.querySelector('.top-right'),
            bottomLeft: DOM.pinContainer.querySelector('.bottom-left'),
            bottomRight: DOM.pinContainer.querySelector('.bottom-right'),
        };
        DOM.trackElement = container.querySelector('.seq-track');
        DOM.trackMarkersContainer = DOM.trackElement?.querySelector('.seq-track-markers');

        timelineDuration = resolveTimelineDuration(options.duration, container);

        prepareImageLayer();
        prepareOverlayCues();
        setupScrollTrigger();

        revealKeyholeMask(75);
        hideKeyholeMask(75, -1);

        observe(container);
    };

    const reset = () => {
        observer?.disconnect();
        scrollTrigger?.kill();

        DOM = {
            container: null,
            pinContainer: null,
            triggerElement: null,
            imageLayer: null,
            overlayLayer: null,
            preloaderLayer: null,
            keyholeMask: null,
            corners: {
                topLeft: null,
                topRight: null,
                bottomLeft: null,
                bottomRight: null,
            },
            trackElement: null,
            trackMarkersContainer: null,
        };

        observer = null;
        scrollTrigger = null;
        lastRenderedTrackHeight = null;

        imageItems = [];
        overlayCueItems = [];

        isScrollLocked = true;
        preloaderRemoved = false;
        isImageUpdateQueued = false;

        scrollDirection = 'down';
        activeImageIndex = -1;
        lastResolvedImageIndex = -1;
        lastScrollProgress = 0;
        queuedImageProgress = 0;
    };

    const setupScrollTrigger = () => {
        if (scrollTrigger || !DOM.trackElement || !DOM.pinContainer) return;

        const getLVH = utils.css.getLVH;

        scrollTrigger = ScrollTrigger.create({
            trigger: DOM.triggerElement,
            pin: DOM.pinContainer,
            start: () => `center ${options.startOffset ?? 0.5 * getLVH()}px`,
            end: () => {
                updateTrack();
                return `+=${DOM.trackElement.offsetHeight - DOM.pinContainer.offsetHeight}`;
            },
            scrub: true,
            pinSpacing: false,
            invalidateOnRefresh: true,
            onUpdate: (self) => {
                if (isScrollLocked) return;

                const progress = self.progress;
                scrollDirection = progress > lastScrollProgress ? 'down' : 'up';
                lastScrollProgress = progress;

                throttledUpdateImageLayer(progress);
                updateOverlayCues(progress);
            }
        });
    };

    // ----- 1 Scroll Coordination -----
    // Controls the entry point and main update cycle.
    // Uses IntersectionObserver to start logic when container enters viewport.

    const observe = (target) => {
        let hasTriggered = false;

        observer = new IntersectionObserver(([entry]) => {
            if (!entry?.isIntersecting || hasTriggered) return;
            hasTriggered = true;
            isScrollLocked = false;

            const matchedIndex = findInitialImageCueIndex();
            const matched = imageItems[matchedIndex];

            // First: preload and activate matched
            preloadImage(matched, (isFirstLoad) => {
                updateImage(matchedIndex, isFirstLoad === false);

                // Step 1: preload surrounding in interleaved order [+1, -1, +2, -2, ...]
                const preloadWindow = 3;
                const preloadOrder = [];
                let i = 1;
                for (; i <= preloadWindow; i++) {
                    if (matchedIndex + i < imageItems.length) preloadOrder.push(matchedIndex + i);
                    if (matchedIndex - i >= 0) preloadOrder.push(matchedIndex - i);
                }

                let j = 0;
                const totalNearby = preloadOrder.length;
                for (; j < totalNearby; j++) {
                    preloadImage(imageItems[preloadOrder[j]], null, false);
                }

                // Step 2: preload *all remaining* images (excluding matched and already preloaded)
                let k = 0;
                const total = imageItems.length;
                for (; k < total; k++) {
                    if (k === matchedIndex || preloadOrder.includes(k)) continue;
                    if (!imageItems[k].loaded) {
                        preloadImage(imageItems[k], null, false);
                    }
                }
            }, true);
        }, {
            threshold: 0.01,
            rootMargin: '0px 0px -0.01% 0px',
        });

        observer.observe(target);
    };

    const throttledUpdateImageLayer = (progress) => {
        queuedImageProgress = progress;
        if (isImageUpdateQueued) return;

        isImageUpdateQueued = true;

        requestAnimationFrame(() => {
            updateImageLayer(queuedImageProgress);
            isImageUpdateQueued = false;
        });
    };

    // ----- 2 Image Layer Preparation -----
    // Parsing DOM and cue data for image items.
    const prepareImageLayer = () => {
        if (!DOM.imageLayer) return;

        const imageEls = DOM.imageLayer.querySelectorAll('.seq-img');
        imageItems.length = 0;

        let imageEl, cueStart, cueEnd;
        let i = 0;
        const total = imageEls.length;

        for (; i < total; i++) {
            imageEl = imageEls[i];
            cueStart = parseFloat(imageEl.getAttribute('data-start'));
            cueEnd = parseFloat(imageEl.getAttribute('data-end'));

            if (!Number.isFinite(cueStart)) continue;
            if (!Number.isFinite(cueEnd)) cueEnd = null;

            gsap.set(imageEl, { opacity: 0 });

            const imageItem = {
                el: imageEl,
                start: cueStart,
                end: cueEnd,
                active: false,
                loaded: false,
                loading: false
            };

            imageItems.push(imageItem);
        }
    };

    // ----- 3 Scroll-to-Cue Resolution -----
    // Maps scroll progress to matched image index.
    const findInitialImageCueIndex = () => {
        const scrollY = window.scrollY;
        const scrollRange = document.documentElement.scrollHeight - window.innerHeight;
        const scrollProgress = scrollRange > 0 ? scrollY / scrollRange : 0;
        const currentTime = +(scrollProgress * timelineDuration).toFixed(4);

        let matchedIndex = -1;
        let i = 0;
        const total = imageItems.length;

        for (; i < total; i++) {
            if (imageItems[i].start <= currentTime) {
                matchedIndex = i;
            } else {
                break;
            }
        }

        return matchedIndex;
    };

    const findMatchingImageIndex = (time) => {
        let i = 0;
        const total = imageItems.length;

        for (; i < total; i++) {
            if (imageItems[i].start > time) return i - 1;
        }
        return total - 1;
    };

    // ----- 4 Image Cue Switching Logic -----
    // Main logic for what to show and how.
    const updateImageLayer = (progress) => {
        const currentTime = +(progress * timelineDuration).toFixed(4);
        const matchedIndex = findMatchingImageIndex(currentTime);

        lastResolvedImageIndex = matchedIndex;

        if (matchedIndex !== -1) {
            activateMatchedImage(matchedIndex);
        } else {
            fadeOutActiveImage();
        }
    };

    const activateMatchedImage = (index) => {
        const matched = imageItems[index];
        const preloadWindow = 2;
        const totalImages = imageItems.length;

        if (!matched.loaded && !matched.loading) {
            preloadImage(matched, () => {
                updateImage(index);
            });
        } else {
            updateImage(index);
        }
    };

    const updateImage = (targetIndex) => {
        if (targetIndex === activeImageIndex) return;

        const prevImageItem = imageItems[activeImageIndex];
        const nextImageItem = imageItems[targetIndex];

        // --- Fade out previous ---
        if (prevImageItem) {
            gsap.killTweensOf(prevImageItem.el);

            if (!preloaderRemoved) {
                // Hard reset without transition
                prevImageItem.el.style.opacity = '0';
                prevImageItem.active = false;
                prevImageItem.el.style.zIndex = '0';
            } else {
                gsap.to(prevImageItem.el, {
                    opacity: 0,
                    duration: 0.5,
                    ease: 'power2.inOut',
                    onComplete: () => {
                        prevImageItem.active = false;
                        prevImageItem.el.style.zIndex = '0';
                    }
                });
            }
        }

        // --- Fade in new ---
        gsap.killTweensOf(nextImageItem.el);
        nextImageItem.active = true;
        nextImageItem.el.style.zIndex = '1';

        if (!preloaderRemoved) {
            nextImageItem.el.style.opacity = '1';

            if (!preloaderRemovalTick) {
                preloaderRemovalTick = utils.system.nextTick(() => {
                    removePreloaderOnce();
                }, null, 250);
            }
        } else {
            gsap.to(nextImageItem.el, {
                opacity: 1,
                duration: 0.8,
                ease: 'power2.out'
            });
        }

        activeImageIndex = targetIndex;
    };

    const fadeOutActiveImage = () => {
        const activeImageItem = imageItems[activeImageIndex];
        if (!activeImageItem) return;

        gsap.killTweensOf(activeImageItem.el);
        gsap.to(activeImageItem.el, {
            opacity: 0,
            duration: 0.5,
            ease: 'power2.inOut',
            onComplete: () => {
                activeImageItem.active = false;
                activeImageItem.el.style.zIndex = '0';
            }
        });

        activeImageIndex = -1;
    };

    // ----- 5 Image Loading -----
    // Handles loading and preload logic.
    const preloadImage = (imageItem, callback = null, shouldUpdateOnLoad = true) => {
        if (imageItem.loaded || imageItem.loading || imageItem.preloadRequested) {
            if (imageItem.loaded && callback && shouldUpdateOnLoad) {
                callback(false); // not first load
            }
            return;
        }

        imageItem.preloadRequested = true;
        imageItem.loading = true;

        const imageEl = imageItem.el.querySelector('img') || imageItem.el;

        if (imageEl.dataset.src) imageEl.src = imageEl.dataset.src;
        if (imageEl.dataset.srcset) imageEl.srcset = imageEl.dataset.srcset;

        const finalize = () => {
            imageItem.loaded = true;
            imageItem.loading = false;
        };

        const done = (isFirstLoad) => {
            finalize();
            if (callback && shouldUpdateOnLoad) {
                callback(isFirstLoad);
            }
        };

        imageEl.onerror = () => done(false);

        if (imageEl.complete && imageEl.naturalWidth !== 0) {
            done(false); // already loaded
        } else {
            imageEl.onload = () => done(true);
            imageEl.onerror = () => done(false);
        }

        //imageEl.decode().then(() => done(true)).catch(() => done(false));
    };

    // ----- 6 Preloader Lifecycle -----
    // Final cleanup after first image has loaded.
    const removePreloaderOnce = () => {
        if (!preloaderRemoved && DOM.preloaderLayer) {
            preloaderRemoved = true;
            DOM.preloaderLayer.style.opacity = '0';
            utils.system.nextTick(() => {
                DOM.preloaderLayer?.parentNode?.removeChild(DOM.preloaderLayer);
                DOM.preloaderLayer = null;
            }, null, 350);
        }
    };

    // ----- 7 Overlay Cue Layer -----
    // Parse overlay cue elements and control visibility transitions during scroll.
    const prepareOverlayCues = () => {
        if (!DOM.overlayLayer) return;

        const cueElements = DOM.overlayLayer.querySelectorAll('.seq-overlay');
        overlayCueItems.length = 0;

        let cueEl, innerEl, start, end;
        let i = 0;
        const total = cueElements.length;

        for (; i < total; i++) {
            cueEl = cueElements[i];
            start = parseFloat(cueEl.getAttribute('data-start'));
            end = parseFloat(cueEl.getAttribute('data-end'));
            innerEl = cueEl.querySelector('.seq-overlay-inner');
            if (!innerEl) continue;

            if (Number.isFinite(start) && Number.isFinite(end)) {
                overlayCueItems.push({
                    container: cueEl,
                    inner: innerEl,
                    start,
                    end,
                    active: false
                });
            }
        }
    };

    const updateOverlayCues = (progress) => {
        const currentTime = progress * timelineDuration;
        const isScrollingDown = scrollDirection === 'down';
        const yIn = isScrollingDown ? 50 : -50;
        const yOut = isScrollingDown ? -50 : 50;

        let activeCue = null;
        let i = 0;
        const total = overlayCueItems.length;
        let cueItem, innerEl, otherCue;

        // Find current active cue
        for (; i < total; i++) {
            cueItem = overlayCueItems[i];
            if (currentTime >= cueItem.start && currentTime < cueItem.end) {
                activeCue = cueItem;
                break;
            }
        }

        // Animate cues
        for (i = 0; i < total; i++) {
            cueItem = overlayCueItems[i];
            innerEl = cueItem.inner;

            if (cueItem === activeCue) {
                if (!cueItem.active) {
                    cueItem.active = true;

                    // Deactivate all others
                    for (let j = 0; j < total; j++) {
                        otherCue = overlayCueItems[j];
                        if (otherCue !== cueItem && otherCue.active) {
                            otherCue.active = false;
                            gsap.killTweensOf(otherCue.inner);
                            gsap.set(otherCue.inner, { opacity: 0, y: 0 });
                        }
                    }

                    gsap.set(innerEl, { opacity: 0, y: yIn });

                    gsap.to(innerEl, {
                        opacity: 1,
                        y: 0,
                        duration: 0.3,
                        ease: 'power2.out',
                        overwrite: true,
                    });
                }
            } else if (cueItem.active) {
                cueItem.active = false;
                gsap.killTweensOf(innerEl);

                gsap.to(innerEl, {
                    opacity: 0,
                    y: yOut,
                    duration: 0.3,
                    ease: 'power1.inOut',
                    overwrite: true,
                    onComplete: () => {
                        gsap.set(innerEl, { y: 0 });
                    },
                });
            }
        }
    };

    // ----- 8 Keyhole Mask Effects -----
    // Scroll-driven clip-path animation that reveals or hides a central "keyhole" mask.
    // Optional decorative corners animate in sync with the mask shape.

    /**
     * Returns the fully expanded polygon shape for the visible keyhole.
     */
    const getKeyholePolygonRevealed = () => [
        '0% 0%', '0% 100%', '0% 100%', '0% 0%',
        '100% 0%', '100% 100%', '0% 100%', '0% 100%',
        '100% 100%', '100% 0%'
    ].join(',');

    /**
     * Returns the collapsed polygon shape for hiding the keyhole.
     * @param {number} innerPercent - The size of the inner remaining keyhole in percent.
     */
    const getKeyholePolygonCollapsed = (innerPercent = 65) => {
        const outer = 100 - innerPercent;
        return [
            '0% 0%', '0% 100%', `${outer}% 100%`, `${outer}% ${outer}%`,
            `${innerPercent}% ${outer}%`, `${innerPercent}% ${innerPercent}%`,
            `${outer}% ${innerPercent}%`, `${outer}% 100%`,
            '100% 100%', '100% 0%'
        ].join(',');
    };

    /**
     * Animates a keyhole mask from one polygon to another, optionally syncing decorative corners.
     * Skips redundant re-animation if clip path has not changed.
     */
    const animateKeyholeClipPath = ({
        maskElement,
        fromPolygon,
        toPolygon,
        scrollTrigger,
        containerElement = null,
        cornerConfigs = null,
        suppressImmediateRender = false
    }) => {
        if (!maskElement || !fromPolygon || !toPolygon || !scrollTrigger) return;

        const clipKey = `${fromPolygon}__${toPolygon}`;
        if (clipKey === lastAnimatedClipPath) return;
        lastAnimatedClipPath = clipKey;

        gsap.fromTo(maskElement, {
            clipPath: `polygon(${fromPolygon})`
        }, {
            clipPath: `polygon(${toPolygon})`,
            ease: 'none',
            scrollTrigger,
            immediateRender: !suppressImmediateRender
        });

        if (containerElement && cornerConfigs) {
            const corners = DOM.corners;

            Object.entries(cornerConfigs).forEach(([key, config]) => {
                const cornerEl = corners[key];
                if (cornerEl && config?.from && config?.to) {
                    gsap.fromTo(cornerEl, config.from, {
                        ...config.to,
                        ease: 'none',
                        scrollTrigger,
                        immediateRender: !suppressImmediateRender
                    });
                }
            });
        }
    };

    /**
     * Scroll-driven keyhole reveal animation (clip-path + corner elements).
     */
    const revealKeyholeMask = (innerPercent = 65, startOffset = 0, endOffset = 0) => {
        if (!DOM.keyholeMask) return;

        const getLVH = utils.css.getLVH;
        const outer = 100 - innerPercent;

        const fromPolygon = getKeyholePolygonCollapsed(innerPercent);
        const toPolygon = getKeyholePolygonRevealed();

        animateKeyholeClipPath({
            maskElement: DOM.keyholeMask,
            fromPolygon,
            toPolygon,
            scrollTrigger: {
                trigger: DOM.pinContainer,
                scrub: true,
                start: () => `top ${getLVH() + startOffset}px`,
                end: () => `center ${0.5 * getLVH() + endOffset}px`,
            },
            containerElement: DOM.pinContainer,
            cornerConfigs: {
                topLeft: { from: { top: `${outer}%`, left: `${outer}%` }, to: { top: '0%', left: '0%' } },
                topRight: { from: { top: `${outer}%`, left: `${innerPercent}%` }, to: { top: '0%', left: '100%' } },
                bottomLeft: { from: { top: `${innerPercent}%`, left: `${outer}%` }, to: { top: '100%', left: '0%' } },
                bottomRight: { from: { top: `${innerPercent}%`, left: `${innerPercent}%` }, to: { top: '100%', left: '100%' } },
            }
        });
    };

    /**
     * Scroll-driven keyhole hide animation (reversed from reveal).
     */
    const hideKeyholeMask = (innerPercent = 65, startOffset = 0, endOffset = 0) => {
        if (!DOM.keyholeMask) return;

        const getLVH = utils.css.getLVH;
        const outer = 100 - innerPercent;

        const fromPolygon = getKeyholePolygonRevealed();
        const toPolygon = getKeyholePolygonCollapsed(innerPercent);

        animateKeyholeClipPath({
            maskElement: DOM.keyholeMask,
            fromPolygon,
            toPolygon,
            scrollTrigger: {
                trigger: DOM.pinContainer,
                scrub: true,
                start: () => `center ${0.5 * getLVH() + startOffset}px`,
                end: () => `bottom ${endOffset}px`,
            },
            containerElement: DOM.pinContainer,
            cornerConfigs: {
                topLeft: { from: { top: '0%', left: '0%' }, to: { top: `${outer}%`, left: `${outer}%` } },
                topRight: { from: { top: '0%', left: '100%' }, to: { top: `${outer}%`, left: `${innerPercent}%` } },
                bottomLeft: { from: { top: '100%', left: '0%' }, to: { top: `${innerPercent}%`, left: `${outer}%` } },
                bottomRight: { from: { top: '100%', left: '100%' }, to: { top: `${innerPercent}%`, left: `${innerPercent}%` } },
            },
            suppressImmediateRender: true
        });
    };

    // ----- 9 Track Rendering -----

    /**
     * Renders vertical markers for time and overlay cue points along the scroll track.
     * Reuses existing elements when possible to avoid unnecessary DOM updates.
     */
    const renderTrackMarkers = () => {
        const container = DOM.trackMarkersContainer;
        if (!DOM.trackElement || !container || overlayCueItems.length === 0) return;

        const trackHeight = DOM.trackElement.offsetHeight;
        const pinHeight = DOM.pinContainer.offsetHeight;
        const scrollableHeight = trackHeight - pinHeight;
        const pinOffset = pinHeight / 2;

        const frag = document.createDocumentFragment();
        const edges = [
            { prop: 'start', className: 'start' },
            { prop: 'end', className: 'end' }
        ];

        // Cache for comparison to avoid DOM churn
        const existing = container.children;
        const expectedCount = Math.floor(timelineDuration) + 1 + overlayCueItems.length * 2;
        if (existing.length !== expectedCount) container.innerHTML = ''; // Mismatch, clear all

        const seconds = Math.floor(timelineDuration);
        let i = 0;
        let progress, offset, marker, existingMarker;

        for (; i <= seconds; i++) {
            progress = i / timelineDuration;
            offset = (progress * scrollableHeight) + pinOffset;

            existingMarker = container.children[i];
            if (existingMarker && existingMarker.classList.contains('seq-track-time-marker')) {
                existingMarker.style.top = `${offset}px`;
                existingMarker.textContent = `${i}s`;
                frag.appendChild(existingMarker);
            } else {
                marker = document.createElement('div');
                marker.className = 'seq-track-time-marker';
                marker.style.top = `${offset}px`;
                marker.textContent = `${i}s`;
                frag.appendChild(marker);
            }
        }

        // Overlay cue markers
        let overlayItem, edge, time, labelEl;
        const offsetIndex = i; // continue from time marker count
        i = 0;
        const total = overlayCueItems.length;

        for (; i < total; i++) {
            overlayItem = overlayCueItems[i];

            for (let j = 0; j < 2; j++) {
                edge = edges[j];
                time = overlayItem[edge.prop];

                progress = time / timelineDuration;
                offset = (progress * scrollableHeight) + pinOffset;

                marker = document.createElement('div');
                marker.className = `seq-track-overlay-marker ${edge.className}`;
                marker.style.top = `${offset}px`;

                labelEl = document.createElement('span');
                labelEl.className = 'seq-track-overlay-label';
                labelEl.textContent = `${time.toFixed(1)}s`;

                marker.appendChild(labelEl);
                frag.appendChild(marker);
            }
        }

        container.innerHTML = '';
        container.appendChild(frag);
    };

    /**
     * Updates track height and offset based on current scrollable content and pin size.
     * Also triggers marker rendering.
     */

    const updateTrack = () => {
        const trackEl = DOM.trackElement;
        const pinEl = DOM.pinContainer;
        if (!trackEl || !pinEl) return;

        const pinHeight = pinEl.offsetHeight;
        const scrollFactor = options.scrollFactor || 1;
        const totalHeight = (pinHeight * timelineDuration) * scrollFactor;
        const newHeight = `${totalHeight - pinHeight}px`;

        if (newHeight === lastRenderedTrackHeight) return;

        trackEl.style.setProperty('--track-offset-y', `-${pinHeight}px`);
        trackEl.style.setProperty('--track-height', newHeight);
        lastRenderedTrackHeight = newHeight;

        renderTrackMarkers();
    };

    const update = () => {
        updateTrack();
    };

    return {
        initialize,
        update,
    };
};

document.addEventListener('DOMContentLoaded', () => {
    if (scroller) scroller.initialize();

    const scrubInstance = createScrub();
    scrubInstance.initialize({
        selector: '#scrub_01',
        scrollFactor: 1.5,
    });

    const globalRefresh = () => {
        scrubInstance.update();
        ScrollTrigger.refresh();
    };

    if (utils.device.isTouch()) {
        window.addEventListener('orientationchange', () => {
            utils.system.nextTick(globalRefresh, null, 500);
        });
    } else {
        window.addEventListener('resize', () => {
            utils.system.nextTick(globalRefresh);
        });
    }

    utils.system.nextTick(() => {
        globalRefresh();
    }, null, 300);

    const isCodePen = document.referrer.includes("codepen.io");
    const hostDomains = isCodePen ? ["codepen.io"] : [];
    hostDomains.push(window.location.hostname);

    const links = document.getElementsByTagName("a");
    utils.url.validateLinks(links, hostDomains);
});