const models = [
  {
    id: "cube",
    name: "Cube",
    details: "Basic cube geometry",
    size: "2.1 KB",
    type: "mesh",
    color: 0x653449,
    geometry: "box"
  },
  {
    id: "sphere",
    name: "Sphere",
    details: "Smooth subdivided sphere",
    size: "3.7 KB",
    type: "mesh",
    color: 0x854d65,
    geometry: "sphere"
  },
  {
    id: "cylinder",
    name: "Cylinder",
    details: "Cylindrical geometry",
    size: "2.8 KB",
    type: "mesh",
    color: 0x7d4158,
    geometry: "cylinder"
  },
  {
    id: "torus",
    name: "Torus",
    details: "Donut-shaped torus",
    size: "4.2 KB",
    type: "mesh",
    color: 0x4d2532,
    geometry: "torus"
  },
  {
    id: "cone",
    name: "Cone",
    details: "Pointed cone",
    size: "1.9 KB",
    type: "mesh",
    color: 0x5a2f40,
    geometry: "cone"
  },
  {
    id: "dodecahedron",
    name: "Dodecahedron",
    details: "Twelve-faced polyhedron",
    size: "5.1 KB",
    type: "mesh",
    color: 0x93627a,
    geometry: "dodecahedron"
  },
  {
    id: "icosahedron",
    name: "Icosahedron",
    details: "Twenty-faced polyhedron",
    size: "4.8 KB",
    type: "mesh",
    color: 0x8b5268,
    geometry: "icosahedron"
  },
  {
    id: "octahedron",
    name: "Octahedron",
    details: "Eight-faced polyhedron",
    size: "3.2 KB",
    type: "mesh",
    color: 0x753d54,
    geometry: "octahedron"
  },
  {
    id: "tetrahedron",
    name: "Tetrahedron",
    details: "Four-faced polyhedron",
    size: "1.5 KB",
    type: "mesh",
    color: 0x4a2838,
    geometry: "tetrahedron"
  },
  {
    id: "torusknot",
    name: "Torus Knot",
    details: "Twisted torus knot",
    size: "8.4 KB",
    type: "mesh",
    color: 0xa06f85,
    geometry: "torusknot"
  },
  {
    id: "prism",
    name: "Prism",
    details: "Octagonal prism",
    size: "3.8 KB",
    type: "mesh",
    color: 0x6d3a4e,
    geometry: "prism"
  },
  {
    id: "lathe",
    name: "Lathe",
    details: "Rotated profile shape",
    size: "5.2 KB",
    type: "mesh",
    color: 0x563045,
    geometry: "lathe"
  }
];

const miniScenes = new Map();
let selectedScene = null;

function createGeometry(type) {
  const geometries = {
    box: () => new THREE.BoxGeometry(1, 1, 1),
    sphere: () => new THREE.SphereGeometry(0.7, 16, 12),
    cylinder: () => new THREE.CylinderGeometry(0.5, 0.5, 1, 12),
    torus: () => new THREE.TorusGeometry(0.5, 0.2, 8, 16),
    cone: () => new THREE.ConeGeometry(0.5, 1, 12),
    dodecahedron: () => new THREE.DodecahedronGeometry(0.7),
    icosahedron: () => new THREE.IcosahedronGeometry(0.7),
    octahedron: () => new THREE.OctahedronGeometry(0.7),
    tetrahedron: () => new THREE.TetrahedronGeometry(0.7),
    torusknot: () => new THREE.TorusKnotGeometry(0.5, 0.2, 64, 8),
    prism: () => new THREE.CylinderGeometry(0.5, 0.5, 1, 8),
    lathe: () => {
      const points = [];
      for (let i = 0; i < 10; i++) {
        points.push(
          new THREE.Vector2(Math.sin(i * 0.2) * 0.3 + 0.3, (i - 5) * 0.1)
        );
      }
      return new THREE.LatheGeometry(points, 12);
    }
  };
  return geometries[type] ? geometries[type]() : geometries.box();
}

function createScene(canvas, model, size = 64) {
  const scene = new THREE.Scene();
  const camera = new THREE.PerspectiveCamera(45, 1, 0.1, 10);
  const renderer = new THREE.WebGLRenderer({
    canvas,
    antialias: true,
    alpha: true
  });

  renderer.setSize(size, size);
  renderer.setClearColor(0x000000, 0);

  const ambientLight = new THREE.AmbientLight(0xffffff, 0.6);
  scene.add(ambientLight);

  const directionalLight = new THREE.DirectionalLight(0xffffff, 0.8);
  directionalLight.position.set(2, 2, 2);
  scene.add(directionalLight);

  const geometry = createGeometry(model.geometry);
  const material = new THREE.MeshLambertMaterial({ color: model.color });
  const mesh = new THREE.Mesh(geometry, material);
  scene.add(mesh);

  camera.position.set(2.5, 1.5, 2.5);
  camera.lookAt(0, 0, 0);

  return { scene, camera, renderer, mesh };
}

function populateSelect() {
  const select = document.getElementById("modelSelect");
  select.innerHTML = "";

  models.forEach((model) => {
    const option = document.createElement("option");
    option.value = model.id;

    const canvas = document.createElement("canvas");
    canvas.className = "model-canvas";
    canvas.width = 64;
    canvas.height = 64;

    option.innerHTML = `
      <div class="model-option">
        <div class="model-preview">${canvas.outerHTML}</div>
        <div class="model-info">
          <div class="model-name">${model.name}</div>
          <div class="model-details">${model.details}</div>
        </div>
        <div class="model-size">${model.size}</div>
        <div class="model-type">${model.type}</div>
      </div>
    `;

    select.appendChild(option);

    // Create scene immediately instead of waiting for requestAnimationFrame
    const actualCanvas = option.querySelector(".model-canvas");
    if (actualCanvas) {
      const sceneData = createScene(actualCanvas, model);
      miniScenes.set(model.id, sceneData);
    }
  });
}

function updateMainPreview(modelId) {
  const model = models.find((m) => m.id === modelId);
  if (!model) return;

  const canvas = document.getElementById("selectedCanvas");
  const info = document.getElementById("selectedInfo");
  const container = canvas.parentElement;

  if (selectedScene) {
    selectedScene.renderer.dispose();
  }

  const size = Math.min(container.clientWidth, container.clientHeight);
  selectedScene = createScene(canvas, model, size);
  info.innerHTML = `<strong>${model.name}</strong> • ${model.details} • ${model.size}`;

  // Update technical details
  updateTechDetails(selectedScene.mesh.geometry);
}

function updateTechDetails(geometry) {
  const vertices = geometry.attributes.position.count;
  const faces = geometry.index ? geometry.index.count / 3 : vertices / 3;

  document.getElementById("vertices").textContent = vertices.toLocaleString();
  document.getElementById("faces").textContent = Math.floor(
    faces
  ).toLocaleString();
}

// Info toggle functionality
document.getElementById("infoToggle").addEventListener("click", function () {
  const details = document.getElementById("techDetails");
  const isHidden = details.hidden;

  details.hidden = !isHidden;
  this.setAttribute(
    "aria-label",
    isHidden ? "Hide technical details" : "Show technical details"
  );
  this.setAttribute(
    "title",
    isHidden ? "Hide technical details" : "Show technical details"
  );
});

function animate() {
  requestAnimationFrame(animate);

  miniScenes.forEach((sceneData) => {
    if (sceneData.mesh) {
      sceneData.mesh.rotation.y += 0.01;
      sceneData.mesh.rotation.x += 0.005;
      sceneData.renderer.render(sceneData.scene, sceneData.camera);
    }
  });

  if (selectedScene?.mesh) {
    selectedScene.mesh.rotation.y += 0.015;
    selectedScene.mesh.rotation.x += 0.008;
    selectedScene.renderer.render(selectedScene.scene, selectedScene.camera);
  }
}

document.getElementById("modelSelect").addEventListener("change", function () {
  if (this.value) {
    updateMainPreview(this.value);
  }
});

populateSelect();
animate();

setTimeout(() => {
  const select = document.getElementById("modelSelect");
  if (models.length > 0) {
    select.value = models[0].id;
    updateMainPreview(models[0].id);
  }
}, 200);