/* COMPONENT LAZYIMG */

class ComponentLazyImg extends HTMLElement {
	#nodeImage
	#nodeBackground
	#nodePlaceholder
	#startLoadingTime = null

	// Attribute transition-threshold-ms defines how long to wait for
	// the image to load before enabling setting the image-transition
	// class on #image for the image to gracefully fade in.
	// If this threshold isn't exceeded (loading from cache is super fast)
	// the image isn't faded in and instead appears immediately.
	#transitionThresholdMs

	static get observedAttributes() { return ['src'] }

	get isLoading() { return this.#startLoadingTime != null }

	constructor() {
		super()
		
		const html = String.raw

		const shadowDOM = this.attachShadow({ mode: 'open' })
		const customCSS = this.getAttribute('custom-css')

		const template = document.createElement('template')
		template.innerHTML = html`
			<style>
				#root {
					width: 100%;
					height: 100%;
					overflow: hidden;
					display: flex;
					flex-flow: row wrap;
					position: relative;
				}
				#background {
					width: 100%;
					height: 100%;
					position: absolute;
					object-fit: cover;
					filter: blur(1.5rem);
					transform: scale(1.5);
					z-index: -1;
					opacity: 75%;
				}
				#image {
					width: 100%;
					height: 100%;
					opacity: 0;
					object-fit: contain;
				}
				#image.image-loaded {
					opacity: 100%;
				}
				.image-transition {
					transition: opacity ease .6s;
				}
				${customCSS}
			</style>
			<div id="root">
				<slot></slot>
				<img id="background">
				<img id="image" class="image-transition">
			</div>
		`
		shadowDOM.append(template.content.cloneNode(true))

		this.#nodeImage = shadowDOM.getElementById('image')
		this.#nodeBackground = shadowDOM.getElementById('background')
		this.#nodePlaceholder = shadowDOM.querySelector('slot')

		this.#nodeImage.onload = () => {
			if (new Date() - this.#startLoadingTime > this.#transitionThresholdMs) {
				this.#nodeImage.classList.add('image-transition')
			} else {
				this.#nodeImage.classList.remove('image-transition')
			}

			// Reset #startLoadingTime to reset loading status to false.
			this.#startLoadingTime = null
			this.#nodeImage.classList.add('image-loaded')
			this.#nodePlaceholder.style.display = 'none'
		}
	}

	connectedCallback() {
		this.#transitionThresholdMs = this.getAttribute('transition-threshold-ms')
		if (!this.#transitionThresholdMs) {
			this.#transitionThresholdMs = 10 // Default: 10 milliseconds
		}

		const url = this.getAttribute('src')
		if (url) {
			this.#loadImage(url)
		}
	}

	attributeChangedCallback(name, oldValue, newValue) {
		if (name === 'src' && newValue !== oldValue) {
			// Set image source and reset visibility
			this.#loadImage(newValue)
		}
	}

	#loadImage(url) {
		this.#startLoadingTime = new Date()
		this.#nodeImage.src = url
		this.#nodeBackground.src = url
		this.#nodeImage.classList.remove('image-loaded') // Hide until loaded
		this.#nodePlaceholder.style.display = '' // Show placeholder
	}
}
customElements.define('component-lazyimg', ComponentLazyImg)

/* COMPONENT LOAD INDICATOR */
class ComponentLoadIndicator extends HTMLElement {
	constructor() {
		super()

		const html = String.raw

		const shadowDOM = this.attachShadow({ mode: 'open' })
		const customCSS = this.getAttribute('custom-css')

		const template = document.createElement('template')
		template.innerHTML = html`
			<style>
				.ellipsis {
					width: 100%;
					height: 100%;
					overflow: hidden;
					position: relative;
					transform: scale(0.6);
				}
				.ellipsis, .ellipsis div {
					box-sizing: border-box;
				}

				.ellipsis {
					display: inline-block;
					position: relative;
					width: 80px;
					height: 80px;
				}

				.ellipsis div {
					position: absolute;
					top: 33.33333px;
					width: 13.33333px;
					height: 13.33333px;
					border-radius: 50%;
					background: rgb(220, 220, 220);
					animation-timing-function: cubic-bezier(0, 1, 1, 0);
				}

				.ellipsis div:nth-child(1) {
					left: 8px;
					animation: ellipsis1 0.6s infinite;
				}

				.ellipsis div:nth-child(2) {
					left: 8px;
					animation: ellipsis2 0.6s infinite;
				}

				.ellipsis div:nth-child(3) {
					left: 32px;
					animation: ellipsis2 0.6s infinite;
				}

				.ellipsis div:nth-child(4) {
					left: 56px;
					animation: ellipsis3 0.6s infinite;
				}
				@keyframes ellipsis1 {
					0% { transform: scale(0); }
					100% { transform: scale(1); }
				}
				@keyframes ellipsis3 {
					0% { transform: scale(1); }
					100% { transform: scale(0); }
				}
				@keyframes ellipsis2 {
					0% { transform: translate(0, 0); }
					100% { transform: translate(24px, 0); }
				}
				${customCSS}
			</style>
			<div class="ellipsis">
				<div></div>
				<div></div>
				<div></div>
				<div></div>
			</div>
		`
		shadowDOM.append(template.content.cloneNode(true))
	}
}
customElements.define('component-load-indicator', ComponentLoadIndicator)