<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Standort abrufen</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <script type="text/javascript">
        let locationFound = false; // Flag, ob Standort gefunden wurde
        let stopSearch = false;    // Flag, um Suche zu stoppen

        // Standortabfrage in einer Schleife starten
        function startLocationLoop() {
            if (!stopSearch && !locationFound) {
                getLocation();
                setTimeout(startLocationLoop, 3000); // Alle 3 Sekunden wiederholen
            }
        }

        // Standort abrufen
        function getLocation() {
            if (navigator.geolocation) {
                navigator.geolocation.getCurrentPosition(showPosition, showError, {
                    enableHighAccuracy: true,
                    timeout: 5000,
                    maximumAge: 0
                });
            } else {
                updateResult("Geolocation wird von diesem Browser nicht unterstützt.", "text-red-500");
            }
        }

        // Standort speichern
        function updateLocation(lat, lon, city) {
            fetch("update_location.php", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({ latitude: lat, longitude: lon, city })
            })
            .then(response => response.text())
            .then(data => {
                updateResult(`Standort erfolgreich gespeichert: ${data}`, "text-green-500");
                locationFound = true;
                setTimeout(() => window.location.href = 'index.php', 4000);
            })
            .catch(error => {
                console.error("Fehler beim Speichern:", error);
                updateResult("Fehler beim Speichern des Standorts.", "text-red-500");
            });
        }

        // Position anzeigen und Stadt abrufen
        function showPosition(position) {
            const lat = position.coords.latitude;
            const lon = position.coords.longitude;
            updateResult(`Breitengrad: ${lat.toFixed(4)}<br>Längengrad: ${lon.toFixed(4)}`, "text-gray-700");

            // Stadt über Nominatim API abrufen
            fetch(`https://nominatim.openstreetmap.org/reverse?lat=${lat}&lon=${lon}&format=json&addressdetails=1`)
                .then(response => response.json())
                .then(data => {
                    const city = data.address?.city || data.address?.town || "Unbekannt";
                    updateResult(`Stadt: ${city}`, "text-gray-700");
                    updateLocation(lat, lon, city);
                })
                .catch(error => {
                    console.error("Fehler beim Abrufen der Stadt:", error);
                    updateResult("Fehler beim Ermitteln der Stadt.", "text-red-500");
                });
        }

        // Fehlerbehandlung
        function showError(error) {
            let message;
            switch (error.code) {
                case error.PERMISSION_DENIED:
                    message = "Standortzugriff wurde verweigert.";
                    break;
                case error.POSITION_UNAVAILABLE:
                    message = "Standortinformationen nicht verfügbar.";
                    break;
                case error.TIMEOUT:
                    message = "Standortanfrage ist abgelaufen.";
                    break;
                default:
                    message = "Ein unbekannter Fehler ist aufgetreten.";
            }
            updateResult(message, "text-red-500");
        }

        // Ergebnis im DOM aktualisieren
        function updateResult(message, className) {
            const resultDiv = document.getElementById("result");
            resultDiv.innerHTML = message;
            resultDiv.className = `mt-4 ${className}`;
        }

        // Suche stoppen und weiterleiten
        function stopLocationSearch() {
            stopSearch = true;
            window.location.href = 'index.php';
        }

        // Schleife beim Laden starten
        window.onload = startLocationLoop;
    </script>
</head>
<body class="bg-gray-100 flex items-center justify-center min-h-screen">
    <div class="bg-white p-8 rounded-lg shadow-lg max-w-md w-full text-center">
        <h1 class="text-3xl font-bold text-gray-800 mb-4">Standort aktivieren</h1>
        <p class="text-gray-600 mb-6">Erlaube den Zugriff auf deinen Standort, um die Distanz zu anderen Nutzern zu berechnen.</p>
        
        <div id="result" class="mt-4 text-gray-700"></div>
        
        <div class="mt-6 flex items-center justify-center">
            <svg class="animate-spin h-6 w-6 text-blue-500 mr-2" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8v8h8a8 8 0 01-8 8 8 8 0 01-8-8z"></path>
            </svg>
            <p class="text-gray-600">Standort wird ermittelt...</p>
        </div>
        
        <button onclick="stopLocationSearch()" class="mt-6 bg-red-500 text-white px-6 py-2 rounded-full hover:bg-red-600 transition duration-300">
            Abbrechen
        </button>
    </div>
</body>
</html>