<?php
// UI only — API in /api
?><!doctype html>
<html lang="de">
<head>
  <meta charset="utf-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>Kommentar-System</title>
  <style>
    :root{
      --bg: #070A12;
      --card: rgba(255,255,255,.06);
      --card2: rgba(255,255,255,.08);
      --stroke: rgba(255,255,255,.12);
      --text: rgba(255,255,255,.92);
      --muted: rgba(255,255,255,.62);
      --accent: #7C5CFF;
      --good: #2EE59D;
      --bad: #FF4D6D;
      --shadow: 0 20px 60px rgba(0,0,0,.45);
      --radius: 18px;
      --mono: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace;
      --sans: ui-sans-serif, system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial, "Apple Color Emoji","Segoe UI Emoji";
    }
    *{box-sizing:border-box}
    body{
      margin:0;
      font-family: var(--sans);
      background: radial-gradient(1200px 600px at 15% 10%, rgba(124,92,255,.22), transparent 60%),
                  radial-gradient(900px 500px at 80% 20%, rgba(46,229,157,.18), transparent 55%),
                  radial-gradient(900px 600px at 70% 90%, rgba(255,77,109,.12), transparent 60%),
                  var(--bg);
      color: var(--text);
      min-height:100vh;
    }
    .wrap{max-width:1000px;margin:0 auto;padding:28px 16px 60px}
    header{
      display:flex;align-items:flex-end;justify-content:space-between;gap:16px;
      padding:10px 0 18px;
    }
    h1{margin:0;font-size:clamp(22px,3vw,34px);letter-spacing:.2px}
    .sub{margin:6px 0 0;color:var(--muted);font-size:14px;line-height:1.5}
    .pill{
      padding:10px 12px;border:1px solid var(--stroke);border-radius:999px;
      background: rgba(255,255,255,.04);
      font-family: var(--mono);
      font-size:12px;color:var(--muted);
      white-space:nowrap;
    }
    .grid{
      display:grid;
      grid-template-columns: 1.05fr .95fr;
      gap:16px;
      align-items:start;
    }
    @media (max-width: 920px){
      .grid{grid-template-columns:1fr}
    }
    .card{
      background: linear-gradient(180deg, var(--card), rgba(255,255,255,.03));
      border: 1px solid var(--stroke);
      border-radius: var(--radius);
      box-shadow: var(--shadow);
      overflow:hidden;
    }
    .card .hd{
      padding:14px 16px;
      display:flex;align-items:center;justify-content:space-between;gap:10px;
      background: linear-gradient(180deg, rgba(255,255,255,.06), rgba(255,255,255,.02));
      border-bottom:1px solid var(--stroke);
    }
    .card .hd b{font-size:14px}
    .card .bd{padding:16px}
    .hint{font-size:13px;color:var(--muted)}
    .comments{
      display:flex;flex-direction:column;gap:10px;
      padding:12px;
    }
    .comment{
      padding:12px 12px;
      border:1px solid var(--stroke);
      background: rgba(255,255,255,.04);
      border-radius:14px;
    }
    .row{
      display:flex;align-items:center;justify-content:space-between;gap:10px;
      margin-bottom:6px;
    }
    .name{
      font-weight:700;font-size:13px;
      display:flex;align-items:center;gap:8px;
    }
    .dot{
      width:8px;height:8px;border-radius:50%;
      background: radial-gradient(circle at 30% 30%, rgba(255,255,255,.9), rgba(255,255,255,.2));
      border:1px solid rgba(255,255,255,.18);
      box-shadow: 0 8px 18px rgba(124,92,255,.25);
    }
    .time{font-size:12px;color:var(--muted);font-family: var(--mono)}
    .msg{white-space:pre-wrap;line-height:1.5;color: rgba(255,255,255,.86);font-size:14px}
    form{display:flex;flex-direction:column;gap:10px}
    .field{display:flex;flex-direction:column;gap:6px}
    label{font-size:12px;color:var(--muted)}
    input, textarea{
      width:100%;
      color:var(--text);
      background: rgba(255,255,255,.04);
      border:1px solid rgba(255,255,255,.12);
      border-radius:14px;
      padding:12px 12px;
      outline:none;
      transition: border-color .2s, transform .05s, background .2s;
      font-family: var(--sans);
      font-size:14px;
    }
    textarea{min-height:140px;resize:vertical}
    input:focus, textarea:focus{
      border-color: rgba(124,92,255,.55);
      background: rgba(255,255,255,.055);
    }
    .actions{display:flex;gap:10px;align-items:center;flex-wrap:wrap;margin-top:4px}
    .btn{
      border:1px solid rgba(255,255,255,.14);
      background: rgba(255,255,255,.06);
      color:var(--text);
      padding:11px 14px;
      border-radius:14px;
      cursor:pointer;
      font-weight:700;
      font-size:14px;
      transition: transform .05s, border-color .2s, background .2s;
      user-select:none;
    }
    .btn:hover{border-color: rgba(124,92,255,.45)}
    .btn:active{transform: translateY(1px)}
    .btn.primary{
      background: linear-gradient(180deg, rgba(124,92,255,.9), rgba(124,92,255,.55));
      border-color: rgba(124,92,255,.35);
      box-shadow: 0 18px 40px rgba(124,92,255,.18);
    }
    .btn.ghost{
      background: transparent;
    }
    .status{
      display:none;
      padding:10px 12px;
      border-radius:14px;
      border:1px solid var(--stroke);
      font-size:13px;
      color:var(--muted);
      background: rgba(255,255,255,.04);
    }
    .status.show{display:block}
    .status.good{
      border-color: rgba(46,229,157,.35);
      color: rgba(46,229,157,.9);
      background: rgba(46,229,157,.08);
    }
    .status.bad{
      border-color: rgba(255,77,109,.35);
      color: rgba(255,77,109,.92);
      background: rgba(255,77,109,.08);
    }
    .foot{
      margin-top:14px;
      display:flex;justify-content:space-between;gap:10px;flex-wrap:wrap;
      color:var(--muted);font-size:12px
    }
    .k{font-family:var(--mono);opacity:.85}
  </style>
</head>
<body>
  <div class="wrap">
    <header>
      <div>
        <h1>Kommentar‑System</h1>
        <div class="sub">Mit Filter gegen unangemessene Wörter (JSON‑Liste) · Modernes UI · AJAX</div>
      </div>
      <div class="pill">/api/list.php · /api/add.php</div>
    </header>

    <div class="grid">
      <section class="card">
        <div class="hd">
          <b>🗨️ Kommentare</b>
          <button class="btn ghost" id="btnReload" type="button">Neu laden</button>
        </div>
        <div class="comments" id="comments"></div>
        <div class="bd">
          <div class="hint">Die neuesten Kommentare werden oben angezeigt.</div>
        </div>
      </section>

      <aside class="card">
        <div class="hd"><b>✍️ Kommentar schreiben</b><span class="hint k" id="charCount">0/2000</span></div>
        <div class="bd">
          <div class="status" id="status"></div>
          <form id="form">
            <div class="field">
              <label for="name">Name</label>
              <input id="name" name="name" maxlength="80" placeholder="Dein Name" autocomplete="name" required>
            </div>
            <div class="field">
              <label for="message">Nachricht</label>
              <textarea id="message" name="message" maxlength="2000" placeholder="Schreib etwas Nettes…" required></textarea>
            </div>
            <div class="actions">
              <button class="btn primary" id="btnSend" type="submit">Senden</button>
              <button class="btn" id="btnClear" type="button">Leeren</button>
            </div>
            <div class="foot">
              <span>Filter‑Liste: <span class="k">forbidden_words.json</span></span>
              <span>DB: <span class="k">comments</span></span>
            </div>
          </form>
        </div>
      </aside>
    </div>
  </div>

<script>
(() => {
  const els = {
    comments: document.getElementById('comments'),
    btnReload: document.getElementById('btnReload'),
    form: document.getElementById('form'),
    name: document.getElementById('name'),
    message: document.getElementById('message'),
    btnSend: document.getElementById('btnSend'),
    btnClear: document.getElementById('btnClear'),
    status: document.getElementById('status'),
    charCount: document.getElementById('charCount'),
  };

  const esc = (s) => (s ?? "").toString()
    .replaceAll("&","&amp;").replaceAll("<","&lt;").replaceAll(">","&gt;")
    .replaceAll('"',"&quot;").replaceAll("'","&#039;");

  function fmtTime(ts){
    try{
      const d = new Date(ts.replace(" ", "T") + "Z"); // MySQL timestamp
      // If server returns local timestamp, this will still be okay-ish.
      return new Intl.DateTimeFormat(undefined, {
        year: "numeric", month: "2-digit", day: "2-digit",
        hour: "2-digit", minute: "2-digit"
      }).format(d);
    }catch(e){
      return ts;
    }
  }

  function showStatus(msg, kind){
    els.status.textContent = msg;
    els.status.className = "status show" + (kind ? (" " + kind) : "");
  }
  function hideStatus(){
    els.status.className = "status";
    els.status.textContent = "";
  }

  async function loadComments(){
    els.btnReload.disabled = true;
    try{
      const res = await fetch("./api/list.php?limit=80", {cache: "no-store"});
      const data = await res.json();
      if(!data.ok) throw new Error("Load failed");
      els.comments.innerHTML = data.comments.map(c => {
        return `
          <div class="comment">
            <div class="row">
              <div class="name"><span class="dot"></span>${esc(c.name)}</div>
              <div class="time">${esc(fmtTime(c.created_at))}</div>
            </div>
            <div class="msg">${esc(c.message)}</div>
          </div>
        `;
      }).join("") || `<div class="comment"><div class="hint">Noch keine Kommentare.</div></div>`;
    }catch(err){
      els.comments.innerHTML = `<div class="comment"><div class="hint" style="color: rgba(255,77,109,.92)">Fehler beim Laden.</div></div>`;
    }finally{
      els.btnReload.disabled = false;
    }
  }

  function updateCount(){
    const len = (els.message.value || "").length;
    els.charCount.textContent = len + "/2000";
  }

  els.btnReload.addEventListener("click", loadComments);
  els.btnClear.addEventListener("click", () => {
    els.name.value = "";
    els.message.value = "";
    updateCount();
    hideStatus();
    els.name.focus();
  });

  els.message.addEventListener("input", () => {
    updateCount();
    hideStatus();
  });
  els.name.addEventListener("input", hideStatus);

  els.form.addEventListener("submit", async (e) => {
    e.preventDefault();
    hideStatus();

    const name = els.name.value.trim();
    const message = els.message.value.trim();

    if(!name || !message){
      showStatus("Bitte Name und Nachricht ausfüllen.", "bad");
      return;
    }

    els.btnSend.disabled = true;
    els.btnSend.textContent = "Sende…";

    try{
      const res = await fetch("./api/add.php", {
        method: "POST",
        headers: {"Content-Type": "application/json"},
        body: JSON.stringify({name, message})
      });
      const data = await res.json().catch(() => ({}));

      if(!res.ok || !data.ok){
        const msg = data.error || "Fehler beim Senden.";
        showStatus(msg, "bad");
        return;
      }

      showStatus("Kommentar gespeichert ✅", "good");
      els.message.value = "";
      updateCount();
      await loadComments();
    }catch(err){
      showStatus("Netzwerkfehler beim Senden.", "bad");
    }finally{
      els.btnSend.disabled = false;
      els.btnSend.textContent = "Senden";
    }
  });

  updateCount();
  loadComments();
})();
</script>
</body>
</html>
